import { useState, useEffect, useCallback } from 'react';

export const useOfflineStorage = () => {
  const [isOnline, setIsOnline] = useState(navigator.onLine);
  const [pendingData, setPendingData] = useState<any[]>([]);

  useEffect(() => {
    const handleOnline = () => setIsOnline(true);
    const handleOffline = () => setIsOnline(false);

    window.addEventListener('online', handleOnline);
    window.addEventListener('offline', handleOffline);

    // Load pending data from localStorage
    const saved = localStorage.getItem('pendingSync');
    if (saved) {
      setPendingData(JSON.parse(saved));
    }

    return () => {
      window.removeEventListener('online', handleOnline);
      window.removeEventListener('offline', handleOffline);
    };
  }, []);

  const addToPendingSync = useCallback((data: any) => {
    const newPending = [...pendingData, { ...data, timestamp: Date.now() }];
    setPendingData(newPending);
    localStorage.setItem('pendingSync', JSON.stringify(newPending));
  }, [pendingData]);

  const syncPendingData = useCallback(async () => {
    if (pendingData.length === 0 || !isOnline) return;

    try {
      // In a real app, this would sync with your PHP backend
      console.log('Syncing pending data:', pendingData);
      
      // Simulate API calls
      for (const item of pendingData) {
        await new Promise(resolve => setTimeout(resolve, 500));
        console.log('Synced:', item);
      }

      // Clear pending data after successful sync
      setPendingData([]);
      localStorage.removeItem('pendingSync');
    } catch (error) {
      console.error('Sync failed:', error);
    }
  }, [pendingData, isOnline]);

  return {
    isOnline,
    pendingData,
    addToPendingSync,
    syncPendingData
  };
};