import React, { useState, useEffect } from 'react';
import { X, Plus, Minus } from 'lucide-react';

interface OrderFormProps {
  onSave: (order: any) => void;
  onCancel: () => void;
}

export const OrderForm: React.FC<OrderFormProps> = ({ onSave, onCancel }) => {
  const [clients, setClients] = useState<any[]>([]);
  const [products, setProducts] = useState<any[]>([]);
  const [formData, setFormData] = useState({
    client_id: '',
    client_name: '',
    items: [] as any[],
    status: 'pending',
  });

  useEffect(() => {
    const savedClients = localStorage.getItem('clients');
    const savedProducts = localStorage.getItem('products');
    
    if (savedClients) setClients(JSON.parse(savedClients));
    if (savedProducts) setProducts(JSON.parse(savedProducts));
  }, []);

  const handleClientChange = (e: React.ChangeEvent<HTMLSelectElement>) => {
    const clientId = e.target.value;
    const client = clients.find(c => c.id.toString() === clientId);
    setFormData(prev => ({
      ...prev,
      client_id: clientId,
      client_name: client?.name || '',
    }));
  };

  const addItem = (product: any) => {
    const existingItem = formData.items.find(item => item.product_id === product.id);
    if (existingItem) {
      setFormData(prev => ({
        ...prev,
        items: prev.items.map(item =>
          item.product_id === product.id
            ? { ...item, quantity: item.quantity + 1 }
            : item
        )
      }));
    } else {
      setFormData(prev => ({
        ...prev,
        items: [...prev.items, {
          product_id: product.id,
          product_name: product.name,
          quantity: 1,
          price: product.price,
        }]
      }));
    }
  };

  const removeItem = (productId: number) => {
    setFormData(prev => ({
      ...prev,
      items: prev.items.filter(item => item.product_id !== productId)
    }));
  };

  const updateQuantity = (productId: number, quantity: number) => {
    if (quantity === 0) {
      removeItem(productId);
    } else {
      setFormData(prev => ({
        ...prev,
        items: prev.items.map(item =>
          item.product_id === productId
            ? { ...item, quantity }
            : item
        )
      }));
    }
  };

  const total = formData.items.reduce((sum, item) => sum + (item.price * item.quantity), 0);

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    onSave({ ...formData, total });
  };

  return (
    <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center p-4 z-50">
      <div className="bg-white rounded-lg shadow-xl w-full max-w-2xl max-h-[90vh] overflow-y-auto">
        <div className="flex items-center justify-between p-6 border-b border-gray-200">
          <h3 className="text-lg font-semibold text-gray-900">Nuevo Pedido</h3>
          <button
            onClick={onCancel}
            className="p-2 hover:bg-gray-100 rounded-full"
          >
            <X className="h-5 w-5 text-gray-500" />
          </button>
        </div>

        <form onSubmit={handleSubmit} className="p-6 space-y-6">
          <div>
            <label className="block text-sm font-medium text-gray-700 mb-2">
              Cliente
            </label>
            <select
              value={formData.client_id}
              onChange={handleClientChange}
              required
              className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-orange-500"
            >
              <option value="">Seleccionar cliente</option>
              {clients.map(client => (
                <option key={client.id} value={client.id}>
                  {client.name} - {client.company}
                </option>
              ))}
            </select>
          </div>

          <div>
            <label className="block text-sm font-medium text-gray-700 mb-2">
              Productos
            </label>
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4 mb-4">
              {products.map(product => (
                <div key={product.id} className="border border-gray-200 rounded-lg p-3">
                  <div className="flex justify-between items-start mb-2">
                    <div>
                      <h4 className="font-medium text-gray-900">{product.name}</h4>
                      <p className="text-sm text-gray-500">${product.price}</p>
                    </div>
                    <button
                      type="button"
                      onClick={() => addItem(product)}
                      className="bg-orange-600 text-white p-1 rounded-full hover:bg-orange-700"
                    >
                      <Plus className="h-4 w-4" />
                    </button>
                  </div>
                </div>
              ))}
            </div>
          </div>

          {formData.items.length > 0 && (
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2">
                Artículos seleccionados
              </label>
              <div className="space-y-2">
                {formData.items.map(item => (
                  <div key={item.product_id} className="flex items-center justify-between bg-gray-50 p-3 rounded-lg">
                    <div>
                      <span className="font-medium">{item.product_name}</span>
                      <span className="text-sm text-gray-500 ml-2">${item.price}</span>
                    </div>
                    <div className="flex items-center space-x-2">
                      <button
                        type="button"
                        onClick={() => updateQuantity(item.product_id, item.quantity - 1)}
                        className="p-1 text-gray-500 hover:text-gray-700"
                      >
                        <Minus className="h-4 w-4" />
                      </button>
                      <span className="w-8 text-center">{item.quantity}</span>
                      <button
                        type="button"
                        onClick={() => updateQuantity(item.product_id, item.quantity + 1)}
                        className="p-1 text-gray-500 hover:text-gray-700"
                      >
                        <Plus className="h-4 w-4" />
                      </button>
                      <button
                        type="button"
                        onClick={() => removeItem(item.product_id)}
                        className="p-1 text-red-500 hover:text-red-700 ml-2"
                      >
                        <X className="h-4 w-4" />
                      </button>
                    </div>
                  </div>
                ))}
              </div>
              <div className="mt-4 pt-4 border-t border-gray-200">
                <div className="flex justify-between items-center">
                  <span className="text-lg font-semibold">Total:</span>
                  <span className="text-lg font-bold text-orange-600">
                    ${total.toLocaleString('es-MX')}
                  </span>
                </div>
              </div>
            </div>
          )}

          <div className="flex space-x-3 pt-4">
            <button
              type="button"
              onClick={onCancel}
              className="flex-1 px-4 py-2 border border-gray-300 text-gray-700 rounded-lg hover:bg-gray-50"
            >
              Cancelar
            </button>
            <button
              type="submit"
              disabled={formData.items.length === 0}
              className="flex-1 px-4 py-2 bg-orange-600 text-white rounded-lg hover:bg-orange-700 disabled:opacity-50 disabled:cursor-not-allowed"
            >
              Crear Pedido
            </button>
          </div>
        </form>
      </div>
    </div>
  );
};