# Sistema de Gestión de Clientes PWA

Una aplicación web progresiva (PWA) para la gestión de clientes, productos y pedidos con capacidades offline.

## Características

- **Gestión de Clientes**: Agregar, editar y buscar clientes
- **Catálogo de Productos**: Administrar productos con precios y stock
- **Pedidos**: Crear y gestionar órdenes de venta
- **Documentos**: Subir y gestionar documentos de clientes (PDFs, imágenes)
- **Offline First**: Funciona sin conexión a internet
- **Autenticación**: Sistema de login seguro
- **Responsive**: Optimizado para móviles y escritorio

## Tecnologías Frontend

- React 18 con TypeScript
- Tailwind CSS para estilos
- Vite para desarrollo y build
- PWA con Service Worker
- LocalStorage para persistencia offline

## Estructura del Backend (PHP + PostgreSQL)

Para implementar el backend completo, necesitarás:

### 1. Base de Datos PostgreSQL

```sql
-- Usuarios
CREATE TABLE users (
    id SERIAL PRIMARY KEY,
    username VARCHAR(50) UNIQUE NOT NULL,
    password_hash VARCHAR(255) NOT NULL,
    name VARCHAR(100) NOT NULL,
    email VARCHAR(100) UNIQUE NOT NULL,
    role VARCHAR(20) DEFAULT 'user',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- Clientes
CREATE TABLE clients (
    id SERIAL PRIMARY KEY,
    name VARCHAR(100) NOT NULL,
    email VARCHAR(100),
    phone VARCHAR(20),
    address TEXT,
    company VARCHAR(100),
    user_id INTEGER REFERENCES users(id),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- Productos
CREATE TABLE products (
    id SERIAL PRIMARY KEY,
    name VARCHAR(100) NOT NULL,
    description TEXT,
    price DECIMAL(10,2) NOT NULL,
    category VARCHAR(50),
    stock INTEGER DEFAULT 0,
    image_url VARCHAR(500),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- Pedidos
CREATE TABLE orders (
    id SERIAL PRIMARY KEY,
    client_id INTEGER REFERENCES clients(id),
    user_id INTEGER REFERENCES users(id),
    total DECIMAL(10,2) NOT NULL,
    status VARCHAR(20) DEFAULT 'pending',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- Elementos del pedido
CREATE TABLE order_items (
    id SERIAL PRIMARY KEY,
    order_id INTEGER REFERENCES orders(id),
    product_id INTEGER REFERENCES products(id),
    quantity INTEGER NOT NULL,
    price DECIMAL(10,2) NOT NULL
);

-- Documentos
CREATE TABLE documents (
    id SERIAL PRIMARY KEY,
    client_id INTEGER REFERENCES clients(id),
    filename VARCHAR(255) NOT NULL,
    file_path VARCHAR(500) NOT NULL,
    file_type VARCHAR(50),
    uploaded_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);
```

### 2. API PHP

Crear endpoints RESTful:

- `POST /api/auth/login` - Autenticación
- `GET /api/clients` - Listar clientes
- `POST /api/clients` - Crear cliente
- `GET /api/products` - Listar productos
- `POST /api/products` - Crear producto
- `GET /api/orders` - Listar pedidos
- `POST /api/orders` - Crear pedido
- `POST /api/upload` - Subir documentos

### 3. Configuración del Servidor

#### Apache .htaccess
```apache
RewriteEngine On
RewriteCond %{REQUEST_FILENAME} !-f
RewriteCond %{REQUEST_FILENAME} !-d
RewriteRule ^api/(.*)$ api/index.php [QSA,L]
```

#### CORS Headers
```php
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS");
header("Access-Control-Allow-Headers: Content-Type, Authorization");
```

## Instalación y Desarrollo

1. Clonar el repositorio
2. Instalar dependencias: `npm install`
3. Ejecutar en desarrollo: `npm run dev`
4. Construir para producción: `npm run build`

## Despliegue

1. Construir la aplicación: `npm run build`
2. Subir archivos de `dist/` a tu servidor web
3. Configurar el backend PHP con PostgreSQL
4. Configurar HTTPS para PWA

## Funcionalidades Offline

- Los datos se guardan en LocalStorage cuando no hay conexión
- Se sincroniza automáticamente cuando vuelve la conexión
- Service Worker cachea recursos estáticos
- Indicador visual del estado de conexión

## Credenciales de Demo

- Usuario: `admin`
- Contraseña: `123456`